define(['angular'], function (angular) {
    'use strict';

    return function (RequestService) {
        var PageableObject = function (fhirResponseObj) {
            fhirResponseObj = fhirResponseObj || {};

            var total,
                links,
                items,
                selfLink;

            angular.extend(this, angular.copy(RequestService), {

                next: function () {
                    var link = this.getNextLink();

                    if (!(link && link.url)) {
                        throw new Error('Next link not found: end of list.');
                    }

                    return this.changePage(link.url);
                },

                prev: function () {
                    var link = this.getPrevLink();

                    if (!(link && link.url)) {
                        throw new Error('Previous link not found: beginning of list.');
                    }

                    return this.changePage(link.url);
                },

                getNextLink: function (linksList) {
                    return this.getLinkByRelation("next", linksList);
                },

                getPrevLink: function (linksList) {
                    return this.getLinkByRelation("previous", linksList);
                },

                getLinkByRelation: function (relation, linksList) {
                    linksList = linksList || links;

                    return this.findInArray(linksList, function (link) {
                        return link.relation && link.relation === relation;
                    });
                },

                changePage: function (url) {
                    if (!url) {
                        throw new Error('URL must be supplied.  Usage: changePage(url)');
                    }

                    return this.sendRequest({
                        method: 'GET',
                        url: url
                    }).then(function (response) {
                        this.update(response);
                        return;
                    }.bind(this));
                },

                update: function (fhirResponseObj) {
                    fhirResponseObj = fhirResponseObj || {};

                    var data = fhirResponseObj.data || {};

                    total = data.total || 0;
                    links = data.link || [];
                    items = this.getResourcesFromList(data.entry || []);
                },

                getAllItems: function (allItems, url) {
                    allItems = allItems || [];
                    url = url || (selfLink ? selfLink.url : '');

                    var numPerPage = 50;

                    return this.sendRequest({
                        method: 'GET',
                        url: url,
                        params: {
                            _count: numPerPage
                        }
                    }).then(function (response) {
                        response = response || {};

                        var data = response.data || {};
                        var currItems = data.entry || [];
                        var linksList = data.link || [];
                        var nextLink = this.getNextLink(linksList) || {};

                        allItems = allItems.concat(currItems);

                        if (nextLink.url) {
                            return this.getAllItems(allItems, nextLink.url);
                        } else {
                            return this.getResourcesFromList(allItems);
                        }
                    }.bind(this));
                },

                getResourcesFromList: function (list) {
                    return list.map(function (item) {
                        return item.resource || item;
                    });
                },

                getItems: function () {
                    return items;
                },

                getTotal: function () {
                    return total;
                },

                getLinks: function () {
                    return links;
                },

                getSelfLink: function () {
                    return selfLink;
                }
            });

            this.update(fhirResponseObj);

            selfLink = this.getLinkByRelation('self') || {};
        };

        return {
            createPageableObject: function (response) {
                return new PageableObject(response);
            }
        }
    };
});